/*
    ym2dma.c

    Converts _unpacked_ YM tunes to a format better suitable for
    CPC+. Supports only YM2 and YM3 types at the moment.

    7/2/2000 Andy Cadley (AndyCadley@bigfoot.com)

    Output format:

    Standard CPC+ DMA list, without an AMSDOS header. Assumes that
    the PPR is set to 0, so that PAUSE n will pause for n scanlines.

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAXPAUSE 311 /* Maximum no. of scanlines to pause per frame */
#define REGS 14

int main(int argc,char *argv[])
{
    FILE *f;
    long length;   
    unsigned char   *data[REGS];  /*  The unpacked YM data    */
    unsigned    current[REGS];
    char    id[5]={0,0,0,0,0};
    int     n,i,changes;
    unsigned long PauseCount;
    
    if (argc != 3)
    {
        printf("Usage: ym2dma source.ym destination.dma\n");
        printf("Raw YM files only. Uncompress with LHA first\n");
        return(EXIT_FAILURE);
    }

    if((f=fopen(argv[1],"rb"))==NULL)
    {
        printf("File open error.\n");
        return(EXIT_FAILURE);
    }

    fseek(f,0,SEEK_END);
    length=ftell(f)-4;
    fseek(f,0,SEEK_SET);

    fread(id,1,4,f);
    if(strcmp(id,"YM2!"))           /*  YM2 is ok   */
        if(strcmp(id,"YM3!"))       /*  YM3 is ok   */
            if(strcmp(id,"YM3b"))   /*  YM3b is ok  */
            {
                printf("Unsupported file format.\n");
                return(EXIT_FAILURE);
            }
            else
            {
                fread(id,1,4,f);    /*  Skip restart for YM3b   */
                length-=4;
            }

    for(n=0;n<REGS;n++)     /*  Allocate memory & read data */
    {

        if((data[n]=(unsigned char*)malloc(length/REGS))==NULL)
        {
            printf("Out of memory.\n");
            exit(EXIT_FAILURE);
        }
        memset(data[n],0,length/REGS);
        fread(data[n],1,length/REGS,f);
    }
    fclose(f); 

    if((f=fopen(argv[2],"wb"))==NULL)
    {
        printf("Cannot open destination file.\n");
        return(EXIT_FAILURE);
    }      

    for(n=0;n<REGS;n++)     /*  Set current values to impossible   */
        current[n]=0xffff;

    PauseCount = MAXPAUSE; /* Inital pause */
    //Write out DMA list
    for(n=0;n<length/REGS;n++)
    {
        changes = 0;
        for(i=0;i<REGS;i++)         /*  ... for each register      */
        {
            if(data[i][n] != current[i])
                if (!((i == 13) && (data[i][n] == 255)))
                    changes++;
        }

        if (changes == 0)
        {
            //Nothing changes so just make last pause longer
            PauseCount = PauseCount + MAXPAUSE + 1;
            if (PauseCount > 4095)
            {
                PauseCount -= 4095;
                fputc(0xff,f);
                fputc(0x1f,f);
            }
        }
        else
        {
            // Need to ensure PAUSE has been written
            if(PauseCount > 0)
            {
                fputc(PauseCount & 0xff,f);
                fputc((0x10)|((PauseCount & 0xf00)>>8),f);
                PauseCount = MAXPAUSE;
            }

            for(i=0;i<REGS;i++)         /*  ... for each register      */
            {
                if(data[i][n] != current[i])
                {
                    if (!((i == 13) && (data[i][n] == 255)))
                    {
                        /* Need to write out a register LOAD instruction */
                        fputc(data[i][n],f);
                        fputc(i,f);

                        current[i] = data[i][n];
                        PauseCount--; /* Each instruction will delay by 64us */
                    }
                }

            }
        }
    }
    /* End DMA list with STOP + INT */
    fputc(0x30,f);
    fputc(0x40,f);

    fclose(f);
    return(EXIT_SUCCESS);
}
